<?php

namespace Incevio\Package\Shippo\Services;

use App\Models\Cart;
use Illuminate\Http\Request;
use Incevio\Package\Shippo\Exceptions\ShippoApiException;
use Log;

class ShippoShippingService
{
    private $order;
    private $sandbox;
    private $address_from;
    private $address_to;
    private $parcel;
    private $shipment;
    private $transaction;
    private $api_token;
    private $base_url;
    private $rate;
    private $label_file_type;

    public function __construct($shop_id)
    {
        $config_shippo = \Incevio\Package\Shippo\Models\ConfigShippo::where('shop_id', $shop_id)->first();
        if(isset($config_shippo))
        {
            $this->sandbox = $config_shippo->sandbox;
            $this->api_token = $config_shippo->merchant_key;
        }
        $this->base_url = config('shippo.base_url');
    }

    /**
     * Create a shipment and get the shipping rates
     *
     * @param array $address_from - The address from which the items will be shipped
     * @param array $address_to - The address to which the items will be shipped
     * @param array $parcel - The parcel containing the items to be shipped
     *
     * @return array - An array containing the shipping rates
     */
    public function createShipment()
    {
        $headers = [
            'Authorization: ShippoToken ' . $this->api_token,
            'Content-Type: application/json'
        ];

        $data = [
            'address_from' => $this->address_from,
            'address_to' => $this->address_to,
            'parcels' => $this->parcel,
            'async' => false
        ];
        $url = $this->base_url . '/shipments/';
        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false // set ssl verification off
        ]);

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            $error_msg = curl_error($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $response = json_decode($response, true);

            Log::error("Error Creating Shipment: $http_code $response");

            throw new ShippoApiException($http_code, $error_msg);
        }

        curl_close($ch);

        return json_decode($response);
    }

    /**
     * Create a transaction and purchase the shipping label
     *
     * @return array - An array containing the transaction details
     */
    public function createTransaction()
    {
        $url = $this->base_url . '/transactions';

        $headers = [
            'Authorization: ShippoToken ' . $this->api_token,
            'Content-Type: application/x-www-form-urlencoded'  // Adjust for form-data
        ];

        $data = [
            'rate' => $this->rate,
            'label_file_type' => $this->label_file_type,
            'async' => false
        ];

        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($data),  // Encode for form-data
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false // set ssl verification off
        ]);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_errno($ch)) {
            $error_msg = curl_error($ch);
        }

        curl_close($ch);

        if ($http_code === 200) {
            return json_decode($response);
        } else {
            Log::error("Error creating Shippo transaction: $http_code $error_msg");

            throw new ShippoApiException($http_code, $error_msg ?? $response);
        }
    }

    /**
     * Create an address in shippo's required format and return the validated address
     * @param array address to validate must contain "country". For more info check https://docs.goshippo.com/shippoapi/public-api/#operation/CreateAddress
     * @return mixed validated address and validation status
     */
    public function createShippoAddress($address_to_verify)
    {
        $url = $this->base_url . '/addresses/';
        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                "Authorization: ShippoToken " . $this->api_token,
                "Content-Type: application/json"
            ],
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($address_to_verify),
            CURLOPT_SSL_VERIFYPEER => false // set ssl verification off
        ]);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            $error_msg = curl_error($ch);
        }

        curl_close($ch);

        if ($http_code === 201) {
            return json_decode($response, true); // Parse response as associative array
        } else {
            Log::error("Error creating Shippo address: $http_code $error_msg $response");

            throw new ShippoApiException($http_code, $error_msg ?? $response);
        }
    }

    /**
     * Get the list of available carrier accounts of the merchant registered in shippo
     *
     * @return array return an array containing carrier details
     */
    public function getAvailableCarrierDetails()
    {
        $ch = curl_init();

        curl_setopt_array($ch, [
            CURLOPT_URL => $this->base_url . '/carrier_accounts?service_levels=true',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                "Authorization: ShippoToken " . $this->api_token
            ],
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_SSL_VERIFYPEER => false // set ssl verification off
        ]);

        $response = curl_exec($ch);

        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            $error_msg = curl_error($ch);
        }

        curl_close($ch);

        if ($http_code === 200) {
            return json_decode($response);
        } else {
            Log::error("Error getting available carrier accounts: $http_code $response");

            throw new ShippoApiException($http_code, $error_msg ?? $response);
        }
    }

    /**
     * Gives live rates for shipping cost based on product information
     * @param string A shippo address object_id can be called with createShippoAddress() 
     * @param array An array of items containing detail of the products to ship. Items will be in json format (enclosed in curly braces '{}')
     * @return array
     */
    public function getLiveRates($address_to, $line_items, $address_from = null)
    {
        $url = $this->base_url . '/live-rates';

        $headers = [
            'Authorization: ShippoToken ' . $this->api_token,
            'Content-Type: application/json',
        ];

        $request_body = [
            "address_to" => $address_to,
            "line_items" => $line_items,
        ];
        if ($address_from) {
            $request_body['address_from'] = $address_to;
        }

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($request_body),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false // set ssl verification off
        ]);

        $response = curl_exec($ch);

        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            $error_msg = curl_error($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            Log::error("Error: Shippo live rates API call failed with message: $http_code $error_msg");

            throw new ShippoApiException($http_code, $error_msg);
        }

        curl_close($ch);

        return json_decode($response);
    }

    /**
     *  Get details of current default parcel
     * @return array returns an associative array containing parcel data
     */
    public function getDefaultParcel()
    {
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->base_url . "/live-rates/settings/parcel-template",
            CURLOPT_HTTPHEADER => [
                "Authorization: ShippoToken " . $this->api_token,
                "Content-Type: application/json"
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_SSL_VERIFYPEER => false // set ssl verification off
        ]);

        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        if (curl_errno($curl)) {
            throw new ShippoApiException($http_code, curl_error($curl));
        }

        curl_close($curl);

        if ($http_code === 200) {
            return json_decode($response);
        } else {
            throw new ShippoApiException($http_code, $response);
        }
    }

    /**
     * Inserts shippo Live rates to the shipping options to show to the customer
     * @param Cart For which shipping rate will be returned
     * @param array containing current shipping options.
     *
     * @return array Updated shipping options that include shipping rates from shippo
     */
    public function insertLiveRatesToFrontend(Cart $cart, $shipping_options)
    {
        $shippo_address = [
            'country' => $cart->Country->iso_code,
            'state' => $cart->State->iso_code
        ];

        $shippo_line_items = [];

        foreach ($cart->inventories as $line_item) {
            $shippo_line_item = [
                "quantity" => $line_item->pivot->quantity,
                "total_price" => number_format($line_item->pivot->unit_price, 2),
                "currency" => get_system_currency(),
                "weight" => "$cart->shipping_weight",
                "weight_unit" => "gm",
                "title" => $line_item->title,
                "manufacture_country" => "US",
                "sku" => $line_item->sku,
            ];

            array_push($shippo_line_items, $shippo_line_item);
        }

        try{
            $response = $this->createShippoAddress($shippo_address);
            $shippo_option = $this->getLiveRates($response['object_id'], $shippo_line_items);
        } catch(\Exception $e){
            Log::error("Error getting live rates: " . $e->getMessage());

            return $shipping_options; // Return original shipping rates to avoid interruption
        }
        
        $option_count = count($shipping_options);

        foreach ($shippo_option->results as $option) {
            $option_count = $option_count + 1;
            
            $shippo_obj = clone $shipping_options->last(); //cloning object to avoid any missing info and format issue
            $shippo_obj->id = $option_count;
            $shippo_obj->name = $option->title;
            $shippo_obj->rate = floatVal($option->amount);
            $shippo_obj->delivery_takes = $option->description;
            
            $shipping_options->push($shippo_obj); // pushing shippo object to add shipping rate at runtime
        }
        
        $shipping_options = $shipping_options->sortby('rate')->values();
        
        return $shipping_options;
    }
}
